/*
 * AbaEngineTest.java  
 *
 * Creator:
 * 21.01.2008 15:14:55 Sippel
 *
 * Maintainer:
 * 21.01.2008 15:14:55 Sippel
 *
 * Last Modification:
 * $Id: $
 *
 * Copyright (c) 2003 ABACUS Research AG, All Rights Reserved
 */
package ch.abacus.testapp;

import ch.abacus.abaengine.*;

import javax.swing.*;
import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.handler.MessageContext;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Timer;
import java.util.TimerTask;
import java.util.Vector;
import java.util.StringTokenizer;
import java.awt.*;

/**
 * This Example demonstrates how an AbaView Report can be started via WebService using the
 * AbaEngine SOAP Interface.  Further Information about the AbaEngine WebService is available
 * via the URL :
 *           http://localhost:40000/av/abaengine/soap/overview.html
 *
 * AbaView Reports must be saved as WebReports (*.avx) if they are to be used with the AbaEngine WebService
 * The *.avx must be present in the x:\abac\kd\abav\web  folder in order to be read via WebService
 *
 * For session debugging the following link to the AbaSioux Status-Page may be useful for developers.
 *        http://localhost:40000/localhost.avxstatus
 * This link only works for HTTP connections (HTTPS is not possible).  The number of open abaengine sessions
 * can be found under the "soap: tokens" title information.
 *
 * The example opens a session and then closes it with each AbaReport request.  When multiple AbaReports are
 * requested they can also be queried via the same session (socket).  One AbaReport can be queried per AbaEngine
 * connection (socket). If the connection is closed, the AbaReport will also be automatically closed/cancelled.
 * If a new Report is opened via the same Connection, any previous Report will be automatically closed.
 *
 * The open SOAP session can be viewed with the URL http://localhost:40000/localhost.avxstatus
 * At the very bottom of the page is a section for "Plugin" with SOAP information e.g. "soap: sessions = 0; time to live = 5 min"
 * If the SOAP Sessions are handled correctly the "session=0" should increment and then decrement (i.e. remain on 0).
 * A maximum of 500 session is allowed, and then further request via AbaEngine WebService will be blocked.
 *
 * AbaView Reports can also be started via the Browser with the URL :
 *           http://localhost:40000/.avx  (no longer possible as of ABACUS V2014)
 *
 * Information : It can be that the User must be defined as an "Internet-Benutzer" in the ABACUS
 * User Management if the User needs to access the AbaReports via the Browser or via WebService.
 *
 * As of ABACUS V2015 it is required that the Mandant Settings are configured to allow "DataAutomation"
 * when using AbaEngine WebService to export data from the Mandant. "DataAutomation" is an ABACUS Licence option.
 *
 * As of ABACUS V2016 the AbaEngine SOAP WebService is not activated by default. The following entries are required in the
 * ABACUS System configuration files to activate the AbaEngine SOAP WebService :
 *  --- ABACUS V2016 and higher ---
 *    \abac\system\abasystem.properties
 *    abaengine.soap.reports = true
 *    abaengine.soap.views = true
 *
 * In most cases, AbaReports are usually exported using the AbaEngine WebService.  If this is the case only the
 * configuration for "abaengine.soap.reports = true" is required.
 *
 * Note : As of ABACUS V2016, the retrieval of the available Report Names with "sys$\\ReportList" is not possible
 *        The Report Name must be known on the server
 *
 * Note : Changes to the ABACUS configuration files generally require a new start of the ABACUS Services.
 *
 */
public class AbaEngineTest extends StandardExampleFrame {

    private int mMaxOpenViewRowCount = 100;  // defines the maximum rows returned per request FindFirst, FindFirstEx and FindNext

    // TESTING : Flag only used for testing to automatically close program after compiling and running
    private int mStartTestSecondsActive = 0;

    public static void main(String[] args) {
        final AbaEngineTest frm = new AbaEngineTest();
        frm.initUI();
        String[] interfaceName = {"abaengine"};
        frm.setInterfaceNames(interfaceName);
        frm.setTitle("AbaEngine / AbaView Example via Web Service JAVA with JAXWS" + ACUtilities.getCurrentProgramVersionInfo());
        frm.setVisible(true);

        if ( args != null ) {
            if ( args.length > 0 ) {
                for ( String param : args ) {
                    if ( param.startsWith("-test") ) {
                        SwingUtilities.invokeLater(new Runnable() {
                            @Override
                            public void run() {
                                frm.mStartTestSecondsActive = 3;
                                frm.setupStartTestTimer();
                            }
                        });
                    }
                }
            }
        }
        // frm.testCsvDecoding();
    }

    /**
     * This method is only used for testing to build, start and close the example application before publication.
     */
    private void setupStartTestTimer() {
        if ( isVisible() && mStartTestSecondsActive > 0 ) {
            final int waitForMilliseconds = mStartTestSecondsActive <= 60 ? (mStartTestSecondsActive * 1000) : mStartTestSecondsActive;
            writeToOutputPane("TESTING : \n\nCurrent Working Directory : " + ACUtilities.getCurrentUserWorkingDirectory() + "\n\nINFO : Program will close automatically in " + String.format("%.1f", (waitForMilliseconds/1000.0)) + " seconds.  (TIME : " + ACUtilities.convertDateToTimeString(System.currentTimeMillis()) + ")");
            // Activate a run once timer - only used to test the Example Programs before publication
            final Timer delayedCloseTimer = new Timer();
            mStartTestSecondsActive = 0;  // So it does not start again
            delayedCloseTimer.schedule(new TimerTask() {
                @Override
                public void run() {
                    System.out.println("   * TIMER EXECUTED : Waiting " + String.format("%.1f", (waitForMilliseconds/1000.0)) + " seconds to close. (TIME : " + ACUtilities.convertDateToTimeString(System.currentTimeMillis()) + ")");
                    delayedCloseTimer.cancel();
                    setVisible(false);
                    dispose();
                }
            }, waitForMilliseconds, (waitForMilliseconds * 100));
        }
    }

    /** JUST FOR TESTING Simple CSV Decoders  */
    private void testCsvDecoding() {
        String[] csvLines = new String[] {
                "ADR_INR,ADR_KURZNA,ADR_NAME,ADR_VORNAME,ADR_ORT,ADR_LAND,ADR_PLZ,ADR_MUTDATUM,ADR_GUID",
                "1,\"Field with, comma\",ABACUS DEMO AG,ABACUS Demo AG,,Wittenbach,CH,9300,21.11.2013 15:12:16,{31142122-620A-DF01-F998-000C2936F612}",
        };

        for ( String csvLine : csvLines ) {
            System.out.println("For CSV : " + csvLine + "");

            String[] elements = decodeCSVLine(9, csvLine, ",");
            System.out.print("1: ");
            for (String elem : elements) {
                System.out.print(" >" + elem + "<  ");
            }
            System.out.println();

            ArrayList<String> values = parseLine(csvLine, DEFAULT_SEPARATOR, DEFAULT_QUOTE);
            System.out.print("2: ");
            for (String elem : values) {
                System.out.print(" >" + elem + "<  ");
            }
            System.out.println();

            values = parseLine2017(csvLine, DEFAULT_SEPARATOR, DEFAULT_QUOTE);
            System.out.print("3: ");
            for (String elem : values) {
                System.out.print(" >" + elem + "<  ");
            }
            System.out.println();
        }
    }

    public void action_getReportNames() {
        // With the Standard AbaView Report "sys$\\ReportList" we can retrieve the
        // Report Names available in x:\abac\kd\abav\web folder
        // Note : As of ABACUS V2016, the retrieval of the available Report Names with "sys$\\ReportList" is not possible
        //        The Report Name must be known on the server
        String reportType = "txt";  // Always use txt Report for report names.
        action_ExportReport(getAbaViewReportNames(), reportType, "", "");
    }

    public void action_runReport() {
        String reportName = getReportName();
        if ( "".equals(reportName) ) {
            showMessageBox("The Report Name is not defined.  Please enter a valid report name", "No Report Name");
            return;
        }

        if ( isQueryAsOpenView() ) {
            //action_OpenView(reportName, getInputParameters());
        } else {
            String reportType = getSelectedAbaViewReportType();
            action_ExportReport(reportName, reportType,  getInputParameters(), getServerExportFileName());
        }
    }


    protected void setupWebServiceUrlBinding( Object wsStubObject ) {

        if (wsStubObject == null) {
            throw new IllegalArgumentException("The WebService Binding Stub Object parameter cannot be null !!");
        }
        BindingProvider bp = (BindingProvider) wsStubObject;
        if (bp == null) {
            throw new IllegalArgumentException("The WebService Binding cannot be cast from Stub Object !!");
        }
        bp.getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, getServiceUrl("abaengine"));

        /*
        Map<String, List<String>> headers = (Map<String, List<String>>) bp.getRequestContext().get(MessageContext.HTTP_RESPONSE_HEADERS);
        if ( headers == null ) {
            if (mUsingGZIPDecompressionForResponse) {
                bp.getRequestContext().put(MessageContext.HTTP_REQUEST_HEADERS, Collections.singletonMap("Accept-Encoding",Collections.singletonList("gzip")));
            }
        } else {
            if (mUsingGZIPDecompressionForResponse) {
                headers.put("Accept-Encoding", Collections.singletonList("gzip"));
            } else if (headers.containsKey("Accept-Encoding")) {
                headers.remove("Accept-Encoding");
            }
        }
        */
//        bp.getRequestContext().put(MessageContext.HTTP_REQUEST_HEADERS, Collections.singletonMap("Accept-Encoding",Collections.singletonList("gzip")));
    }


    @SuppressWarnings("unchecked")
    private boolean action_GetAbaViewReport(String abaViewReportName, String reportType, String abaViewInputParameters)
    {
        boolean reportOk = false;
        Cursor currentCursor = getCursor();
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        getTableHeaderData().clear();
        getTableRowData().clear();
        clearInfo();
        long totalRowCount = 0;
        long startAbaReportRequest = System.currentTimeMillis();

        boolean logSoapMessages = isLogSoapMessagesActive();
        String soapMessagesFileName = ACUtilities.getCurrentUserWorkingDirectory() + File.separator + "temp_SoapMessages.log";
        File soapMessageFile = new File(soapMessagesFileName);
        if ( soapMessageFile.exists() ) {
            if ( ! soapMessageFile.delete() ) {
                writeToOutputPane("Warning : Soap Messaeg log file " + soapMessagesFileName + "could not be deleted" );
            }
        }

        int timeoutMillisecs = getTimeoutMillisecs();
        writeToOutputPane("Info : Using timeout " + timeoutMillisecs + " ms" );
        writeToOutputPane("Starting query for report " + abaViewReportName + " in Mandant " + getCurrentMandant() );
        try
        {
            File wsdlFile = new File("e:\\dev\\abaengine_jaxws_2018_runReportOnly\\wsdl_runReportOnly\\abaengine_runReportOnly.wsdl");
            AbaEngineService service = new AbaEngineService(wsdlFile.toURI().toURL(), new QName("urn:AbaEngine", "AbaEngineService"));

            AbaEngine eng = service.getAbaEnginePort();

            setupWebServiceUrlBinding(eng);

            String abaToken = eng.login(getCurrentUserName(), getCurrentUserPassword());
            if ( abaToken == null || "".equals(abaToken) ) {
                writeToOutputPane("Login unsuccessful" );
                setCursor(currentCursor);
                return reportOk;
            }
            writeToOutputPane("Login successful  Token[" + abaToken + "]" );

            String reportTypeParam = reportType;
            if (getReportSupportsEncoding(reportTypeParam, "UTF-8"))
            {
                reportTypeParam += "/UTF-8";   // Add the encoding for text and html report files
            }
            String returnedData = "";
            if ( !ACUtilities.StringIsNullOrEmpty(abaViewReportName.trim()) ) {

                writeToOutputPane("Info: Starting Query for Report[" + abaViewReportName + "]  " + (ACUtilities.StringIsNullOrEmpty(abaViewInputParameters) ? "" : ("InputParams[" + abaViewInputParameters + "]")) + "");
                ReportResult repResult = eng.runReport(abaToken,
                        abaViewReportName,      // Report = \abac\kd\abav\web\adr.avx
                        reportTypeParam,        // Text-Ausgabe, UTF-8
                        (abaViewInputParameters == null ? "" : abaViewInputParameters), //"parameter1=3;parameter2=99", // Report Eingabewerte
                        "de",                   // Sprache = deutsch
                        getCurrentMandant(),    // Mandant = 7777
                        0,                      // Jahr = 2017, 0 is the actual year
                        timeoutMillisecs);      // Timeout = 0.5 Sekunden

                int timeoutLoopCount = 0;
                while ("timeout".equals(repResult.getStatus())) {
                    timeoutLoopCount++;
                    writeToOutputPane("    Timeout loop [" + timeoutLoopCount + "]  Session[" + repResult.getSession() + "]   [" + (System.currentTimeMillis() - startAbaReportRequest) + " ms]");

                    repResult = eng.retry(repResult.getSession(), timeoutMillisecs);  // Timeout = 0.5 Sekunden
                }
                String suffixInfo = "ok".equalsIgnoreCase(repResult.getStatus()) ? "   Report finished - and being processed..." : "";
                writeToOutputPane("Total Timeout loops [" + timeoutLoopCount + "]" + suffixInfo);

                String fileExtn = getReportFileExtension(reportType);
                String localFileName = ACUtilities.getCurrentUserWorkingDirectory() + File.separator + "temp_abaViewReport." + fileExtn;
                boolean reportFileSaved = false;

                returnedData = (repResult.getData() == null ? "": new String(repResult.getData(), "UTF-8"));

                if ("ok".equals(repResult.getStatus())) {
                    // Save the returned bytes in a local file
                    try {
                        FileOutputStream fo = new FileOutputStream(localFileName);
                        fo.write(repResult.getData());
                        fo.close();
                        reportFileSaved = new File(localFileName).exists();
                        writeToOutputPane("Report data saved to file : " + localFileName);
                    } catch (IOException e) {
                        System.out.println("Exception : " + e.getMessage());
                        e.printStackTrace();
                    }

                    if ("txt".equals(reportType)) {

                        writeToOutputPane("Retrieving report data...");
                        StringTokenizer st = new StringTokenizer(returnedData, "\r\n");
                        int expectedCount = 0;
                        while (st.hasMoreTokens()) {
                            String reportLine = st.nextToken();
                            //                    System.out.println("Report Line [" + reportLine + "]");
                            String[] elements = decodeCSVLine(expectedCount, reportLine, ",");
                            if (expectedCount <= 0) {
                                expectedCount = elements.length;
                                for (String headerText : elements) {
                                    getTableHeaderData().add(removeQuotes(headerText));
                                }
                            } else {
                                Vector rowData = new Vector();
                                for (String elementText : elements) {
                                    rowData.add(removeQuotes(elementText));
                                }
                                getTableRowData().add(rowData);
                                totalRowCount++;
                            }
                        }
                        writeToOutputPane("Number of returned rows in report [" + getTableRowData().size() + "]");
                    } else {
                        getTableHeaderData().add("Information");
                        Vector rowData = new Vector();
                        rowData.add("INFO : Report with type \"" + reportType + "\" will not be shown as table data.");
                        getTableRowData().add(rowData);
                        if (reportFileSaved) {
                            rowData = new Vector();
                            rowData.add("Report file saved as : " + localFileName);
                            getTableRowData().add(rowData);
                        }

                        writeToOutputPane("Report data is only shown in the above window for TEXT format.");
                    }
                    reportOk = true;
                } else {
                    writeToOutputPane("Ein Fehler ist aufgetreten: ");
                    // The Report errors can be returned in the DATA as Bin64 encoded - so decode it
                    if (!ACUtilities.StringIsNullOrEmpty(returnedData)) {
                        writeToOutputPane("------------------------------------");
                        writeToOutputPane(returnedData);
                        writeToOutputPane("------------------------------------");
                    }
                    if (abaViewReportName.equals(getAbaViewReportNames())) {
                        writeToOutputPane("Note : In ABACUS V2016 and later it is not possible to query the report names list with : " + getAbaViewReportNames());
                        writeToOutputPane("The name of the report on the server must be known.");
                    }
                }

                if (abaViewReportName.equals(getAbaViewReportNames())) {
                    setAbaViewReportNamesActive((getTableRowData().size() > 0));
                } else {
                    setAbaViewReportNamesActive(false);
                }

                // nach 15 Minuten beendet der Server die Session, wenn nicht von Zeit zu Zeit "keepAlive" aufgerufen wird
                if (!"".equals(repResult.getSession())) {
                    KeepAliveResult kar = eng.keepAlive(repResult.getSession());
                    if (kar.getTimeToLive() == 0 || "".equals(kar.getSession())) {
                        writeToOutputPane("The session is finished.");
                    }
                    boolean sessionCloseOk = eng.close(repResult.getSession());
                    writeToOutputPane("Session Close returned [" + (sessionCloseOk ? "TRUE" : "FALSE") + "]");
                }
            }
            boolean logoutOk = eng.logout(abaToken);
            writeToOutputPane("Logout - returned [" + (logoutOk ? "TRUE":"FALSE") + "]  LoginToken[" + abaToken + "]");

            String soapMessages = "";  // eng.getSoapLoggingMessages();
            if ( soapMessages != null && !"".equals(soapMessages) ) {
                // Save the SOAP Messages in a local file
                try {
                    String encoding = "UTF-8";
                    FileOutputStream fo = new FileOutputStream(soapMessagesFileName);
                    fo.write(soapMessages.getBytes(encoding));
                    fo.write("\n\n".getBytes(encoding));
                    fo.write("Returned Report Data (decoded)\n==============================================\n".getBytes(encoding));
                    fo.write(returnedData.getBytes(encoding));
                    fo.write("\n----------------------------------------------\n".getBytes(encoding));
                    fo.close();
                    writeToOutputPane("SOAP Messages saved to file : " + soapMessagesFileName );
                } catch (IOException e) {
                    System.out.println("Exception : " + e.getMessage() );
                    e.printStackTrace();
                }
            }
        }
        catch (Exception ex)
        {
            writeToOutputPane("ERROR : An Exception occurred will accessing the report !");
            writeToOutputPane("Exception : " + ex.getMessage() );
            ex.printStackTrace();
        }
        setCursor(currentCursor);

        long endAbaReportRequest = System.currentTimeMillis();
        writeToOutputPane(getTimingReport("AbaReport " + reportType + " Timing Report", startAbaReportRequest, endAbaReportRequest, totalRowCount ));

        refreshReportTable();
        scrollOutputPaneToStart();
        return reportOk;
    }

    private void action_ExportReport(String abaViewReportName, String reportType, String abaViewInputParameters, String exportFilename) {
        if ( abaViewReportName == null || "".equals(abaViewReportName) ) return;
        boolean useOpenView = isQueryAsOpenView() && !abaViewReportName.equals(getAbaViewReportNames());
        boolean reportOk = false;
        if ( ! isShowLoginOnceSelected() || getLoginCount() == 0 ) {
            LoginDialog loginDialog = new LoginDialog(this);
            loginDialog.setUserName(getCurrentUserName());
            loginDialog.setPassword(getCurrentUserPassword());
            loginDialog.setMandant(getCurrentMandant());
            loginDialog.setLocation(getX() + 100, getY() + 50);
            loginDialog.setModal(true);
            loginDialog.setVisible(true);

            if ( !loginDialog.isDialogEndedWithOK() ) {
                return;
            }
            setCurrentUserName(loginDialog.getUserName());
            setCurrentUserPassword(loginDialog.getPassword());
            setCurrentMandant(loginDialog.getMandant());
            incrementLoginCount();

            if ( useOpenView ) {
            } else {
                reportOk = action_GetAbaViewReport(abaViewReportName, reportType, abaViewInputParameters);
                if (reportOk) {
                    action_ExportAbaViewReportToServerFile(abaViewReportName, reportType, abaViewInputParameters, exportFilename);
                }
            }
        } else {
            if ( useOpenView ) {
            } else {
                reportOk = action_GetAbaViewReport(abaViewReportName, reportType, abaViewInputParameters);
                if (reportOk) {
                    action_ExportAbaViewReportToServerFile(abaViewReportName, reportType, abaViewInputParameters, exportFilename);
                }
            }
        }
        if ( reportOk ) {
            addReportName(abaViewReportName);
        }
    }

    private boolean action_ExportAbaViewReportToServerFile(String abaViewReportName, String reportType, String abaViewInputParameters, String exportFilename)
    {
        boolean reportOk = false;
        if ( exportFilename == null || "".equals(exportFilename)) return reportOk;  // No export file defined
        Cursor currentCursor = getCursor();
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

        int timeoutMillisecs = getTimeoutMillisecs();
        writeToOutputPane("Info : Using timeout " + timeoutMillisecs + " ms" );

        writeToOutputPane("Exporting Report to file : " + exportFilename );
        writeToOutputPane("Starting query for export report " + abaViewReportName + " in Mandant " + getCurrentMandant() );

        boolean logSoapMessages = isLogSoapMessagesActive();
        String soapMessagesFileName = ACUtilities.getCurrentUserWorkingDirectory() + File.separator + "temp_SoapMessages.log";
        File soapMessageFile = new File(soapMessagesFileName);
        if ( soapMessageFile.exists() ) {
            if ( ! soapMessageFile.delete() ) {
                writeToOutputPane("Warning : Soap Messaeg log file " + soapMessagesFileName + "could not be deleted" );
            }
        }
        String returnedData = "";
        try
        {
            AbaEngineService service = new AbaEngineService(new URL("e:\\dev\\abaengine_jaxws_2018_runReportOnly\\wsdl_runReportOnly\\abaengine_runReportOnly.wsdl"), new QName("http://www.abacus.ch/abaconnect/" + "AbaEngine", "AbaEngine"));

            AbaEngine eng = service.getAbaEnginePort();

            String abaToken = eng.login(getCurrentUserName(), getCurrentUserPassword());
            if ( abaToken == null || "".equals(abaToken) ) {
                writeToOutputPane("Login unsuccessful" );
                setCursor(currentCursor);
                return reportOk;
            }
            writeToOutputPane("Login successful  Token[" + abaToken + "]" );

            String reportTypeParam = reportType;
            if (getReportSupportsEncoding(reportTypeParam, "UTF-8"))
            {
                reportTypeParam += "/UTF-8";   // Add the encoding for text and html report files
            }

            if ( !ACUtilities.StringIsNullOrEmpty(abaViewReportName.trim()) ) {
                ReportResult fileRepResult = eng.exportToFile(abaToken,
                        abaViewReportName,      // Report = \abac\kd\abav\web\adr.avx
                        exportFilename,
                        reportTypeParam,        // Text-Ausgabe, UTF-8
                        (abaViewInputParameters == null ? "" : abaViewInputParameters), //"parameter1=3|parameter2=99", // Report Eingabewerte
                        "de",                   // Sprache = deutsch
                        getCurrentMandant(),    // Mandant = 7777
                        0,                      // Jahr = 2017, 0 is the actual year
                        timeoutMillisecs);      // Timeout = 0.5 Sekunden

                int timeoutLoopCount = 0;
                while ("timeout".equals(fileRepResult.getStatus())) {
                    fileRepResult = eng.retry(fileRepResult.getSession(), timeoutMillisecs);  // Timeout = 0.5 Sekunden
                    timeoutLoopCount++;
                    writeToOutputPane("    Timeout loop [" + timeoutLoopCount + "]");
                }
                String suffixInfo = "ok".equalsIgnoreCase(fileRepResult.getStatus()) ? "   Report finished." : "";
                writeToOutputPane("Total Timeout loops [" + timeoutLoopCount + "]" + suffixInfo);

                returnedData = (fileRepResult.getData() == null ? "": new String(fileRepResult.getData(), "UTF-8"));
                if ("ok".equals(fileRepResult.getStatus())) {
                    writeToOutputPane("The AbaView Report was successfully exported to the file : " + exportFilename);
                    reportOk = true;
                    if (!exportFilename.contains(":")) {
                        writeToOutputPane("Info : Export file name [" + exportFilename + "] has no path.  Default path on server");
                        writeToOutputPane("          may be a Window System directory (e.g. c:\\windows\\system32\\" + exportFilename + ")");
                    }
                } else {
                    writeToOutputPane("ERROR: An error occurred exporting the AbaView Report to the file : " + exportFilename);
                }

                // nach 15 Minuten beendet der Server die Session, wenn nicht von Zeit zu Zeit "keepAlive" aufgerufen wird
                if (!"".equals(fileRepResult.getSession())) {
                    KeepAliveResult kar = eng.keepAlive(fileRepResult.getSession());
                    if (kar.getTimeToLive() == 0 || "".equals(kar.getSession())) {
                        writeToOutputPane("The session is finished.");
                    }
                    boolean sessionCloseOk = eng.close(fileRepResult.getSession());
                    writeToOutputPane("Session Close returned [" + (sessionCloseOk ? "TRUE" : "FALSE") + "]");
                }
            } else {
                writeToOutputPane("Info : No Report requested, because the the Report Name is not defined.");
                writeToOutputPane("Info : Only the login and logout functionality was tested.");
            }
            boolean logoutOk = eng.logout(abaToken);
            writeToOutputPane("Logout - returned [" + (logoutOk ? "TRUE":"FALSE") + "]  LoginToken[" + abaToken + "]");

            String soapMessages = "";
            if ( soapMessages != null && !"".equals(soapMessages) ) {
                // Save the SOAP Messages in a local file
                try {
                    String encoding = "UTF-8";
                    FileOutputStream fo = new FileOutputStream(soapMessagesFileName);
                    fo.write(soapMessages.getBytes("UTF-8"));
                    fo.write("\n\n".getBytes());
                    fo.write("Returned Report Data (decoded)\n==============================================\n".getBytes());
                    fo.write(returnedData.getBytes());
                    fo.write("\n----------------------------------------------\n".getBytes());
                    fo.close();
                    writeToOutputPane("SOAP Messages saved to file : " + soapMessagesFileName );
                } catch (IOException e) {
                    System.out.println("Exception : " + e.getMessage() );
                    e.printStackTrace();
                }
            }
        }
        catch (Exception ex)
        {
            ex.printStackTrace();
        }
        setCursor(currentCursor);
        return reportOk;
    }

    private static final char DEFAULT_SEPARATOR = ',';
    private static final char DEFAULT_QUOTE = '"';

    public static ArrayList<String> parseLine2017(String cvsLine, char separator, char customQuote) {
        ArrayList<String> fields = new ArrayList<String>();
        if (cvsLine == null && cvsLine.isEmpty()) return fields;

        String fldSeparator = String.valueOf(separator);
        String quoteDelimeter = String.valueOf(customQuote);

        int iposSeparator = 0;
        int iposLastSeparator = 0;
        int iposStartQuote;

        iposSeparator = cvsLine.indexOf(fldSeparator,iposSeparator);
        while ( iposSeparator >= 0 ) {

            if ( iposLastSeparator <= iposSeparator ) {
                String value = ( iposSeparator-iposLastSeparator > 0 ) ? cvsLine.substring(iposLastSeparator, iposSeparator) : "";

                iposStartQuote = value.trim().startsWith(quoteDelimeter) ? cvsLine.indexOf(quoteDelimeter,iposLastSeparator) : -1;
                if ( iposStartQuote >= 0 ) {
                    int iposEndQuote = cvsLine.indexOf(quoteDelimeter,iposStartQuote+1);
                    if ( iposEndQuote > iposStartQuote ) {
                        value = cvsLine.substring(iposStartQuote, iposEndQuote+1);
                        iposSeparator = cvsLine.indexOf(fldSeparator,iposEndQuote); // Move current separtor to next one ofter end Quote
                    } else {
                        value = cvsLine.substring(iposStartQuote);  // No closing Quote found - take the remaining string - this could be an error in CSV format
                        iposSeparator = -1;
                    }
                    fields.add(value);
                } else {
                    fields.add(value);
                }
            }
            if ( iposSeparator >= 0 ) {
                iposLastSeparator = iposSeparator + 1;
                iposSeparator = cvsLine.indexOf(fldSeparator, iposLastSeparator);
            }
        }

        iposSeparator = cvsLine.length() - 1;
        if ( iposLastSeparator < iposSeparator && iposSeparator-iposLastSeparator > 0 ) {
            String value = ( iposSeparator-iposLastSeparator > 0 ) ? cvsLine.substring(iposLastSeparator) : "";
            fields.add(value);
        }

        return fields;
    }

    /*  Method does not work correctly for quoted values - end quote is missing */
    public static ArrayList<String> parseLine(String cvsLine, char separator, char customQuote) {

        ArrayList<String> result = new ArrayList<String>();

        //if empty, return!
        if (cvsLine == null && cvsLine.isEmpty()) return result;

        if (customQuote == ' ') {
            customQuote = DEFAULT_QUOTE;
        }

        if (separator == ' ') {
            separator = DEFAULT_SEPARATOR;
        }

        StringBuilder curVal = new StringBuilder();
        boolean inQuotes = false;
        boolean startCollectChar = false;
        boolean doubleQuotesInColumn = false;

        char[] chars = cvsLine.toCharArray();

        for (char ch : chars) {

            if (inQuotes) {
                startCollectChar = true;
                if (ch == customQuote) {
                    inQuotes = false;
                    doubleQuotesInColumn = false;
                } else {

                    //Fixed : allow "" in custom quote enclosed
                    if (ch == '\"') {
                        if (!doubleQuotesInColumn) {
                            curVal.append(ch);
                            doubleQuotesInColumn = true;
                        }
                    } else {
                        curVal.append(ch);
                    }

                }
            } else {
                if (ch == customQuote) {

                    inQuotes = true;

                    //Fixed : allow "" in empty quote enclosed
                    if (chars[0] != '"' && customQuote == '\"') {
                        curVal.append('"');
                    }

                    //double quotes in column will hit this!
                    if (startCollectChar) {
                        curVal.append('"');
                    }

                } else if (ch == separator) {

                    result.add(curVal.toString());

                    //curVal = new StringBuilder();
                    curVal.setLength(0);
                    startCollectChar = false;

                } else if (ch == '\r') {
                    //ignore LF characters
                    continue;
                } else if (ch == '\n') {
                    //the end, break!
                    break;
                } else {
                    curVal.append(ch);
                }
            }
        }
        result.add(curVal.toString());
        return result;
    }


    /**
     * Decodes the fields in a comma separated line format.  The delimeter characters used to separate the fields
     * and mark text fields can be specified.
     *
     * @param csvLine the comma-separated line to be decoded
     * @param field_delimeter the fields delimeter used on the csvLine e.g. comma character
     * @param text_delimeter the text delimeter used on the csvLine e.g. quote character
     * @return returns an array list of the fields in the csvLine
     */
    static private ArrayList<String> decodeCsvLine(String csvLine, String field_delimeter, String text_delimeter) {
        ArrayList<String> values = new ArrayList<String>();
        if ( csvLine == null || "".equals(csvLine) ) return values;

        boolean quotesPresent;
        String fieldValue = "";
        boolean fieldValueSet = false;
        int sequentialFieldDelimeterCount = 0;
        boolean isFieldDelimeter = false;
        boolean isFieldValueEmpty = true;
        StringTokenizer rangeTokens = new StringTokenizer(csvLine, field_delimeter, true);
        quotesPresent = csvLine.contains(text_delimeter);
        while (rangeTokens.hasMoreTokens()){
            String tokenValue = rangeTokens.nextToken();
            isFieldDelimeter = field_delimeter.equals(tokenValue);
            isFieldValueEmpty = "".equals(fieldValue);

            if ( isFieldDelimeter && isFieldValueEmpty ) {
                if ( sequentialFieldDelimeterCount > 0 ) {
                    fieldValueSet = true;
                }
                sequentialFieldDelimeterCount++;
            } else {
                sequentialFieldDelimeterCount = 0;
                fieldValue += tokenValue;
                fieldValueSet = !quotesPresent || !fieldValue.startsWith(text_delimeter) || fieldValue.endsWith(text_delimeter);
            }

            if ( fieldValueSet ) {
                if ( fieldValue.length() > 1 && fieldValue.startsWith(text_delimeter) && fieldValue.endsWith(text_delimeter) ) {
                    fieldValue = fieldValue.substring(1,(fieldValue.length()-1));
                }
                values.add(fieldValue);
                fieldValue = "";
                fieldValueSet = false;
            }
        }
        if ( sequentialFieldDelimeterCount > 0 && isFieldDelimeter && isFieldValueEmpty ) {
            // Add empty field for last empty delimeter
            values.add("");
        }
        return values;
    }

    // A very basic and slow routine to decode comma separated values with additional commas in the text fields
    static private String[] decodeCSVLine(int expectedCount, String csvLine, String separators)
    {
        ArrayList<String> fieldsList = decodeCsvLine(csvLine,separators,"\"");
        if ( (expectedCount <= 0 && fieldsList.size() > 0) || fieldsList.size() == expectedCount ) {
            return fieldsList.toArray(new String[fieldsList.size()]);
        }
        Vector<String> values = new Vector<String>();
        StringTokenizer st = new StringTokenizer(csvLine,separators);
        while (st.hasMoreTokens() ) {
            String element = st.nextToken();
            values.add(element);
        }

        boolean keepAdding;
        String[] elements = values.toArray(new String[]{});
        if (expectedCount > 0 && expectedCount != elements.length)
        {
            System.out.println("  ***  decodeCSVLine : comma separation correction for : " + csvLine);
//                debugOutputList(elements);
            for (int index = 0; index < (elements.length - 1); index++)
            {
                if (elements[index].startsWith("\"") && !elements[index].endsWith("\""))
                {
                    keepAdding = true;
                    for( int nextIndex = index; nextIndex < (elements.length - 1) && keepAdding; nextIndex++  )
                    {
                        elements[index] += ",";
                        elements[index] += elements[index + 1];
                        for (int nn = (index + 1); nn < elements.length; nn++)
                        {
                            if ((nn + 1) < elements.length)
                            {
                                elements[nn] = elements[(nn + 1)];
                            }
                            else
                            {
                                elements[nn] = "";
                            }
                        }
                        keepAdding = !elements[index].endsWith("\"");

//                        System.out.print("Elements : ");
//                        for (int pp = 0; pp < elements.length; pp++) {
//                            if ( pp > 0 ) System.out.print(",");
//                            System.out.print("[");
//                            System.out.print(elements[pp]);
//                            System.out.print("]");
//                        }
//                        System.out.println("");
                    }
                }
            }
        }
        return elements;
    }


    // A very basic and slow routine to decode comma separated values with additional commas in the text fields
    static private String[] decodeCSVLine_EmptyFieldProblem(int expectedCount, String csvLine, String separators)
    {
        Vector<String> values = new Vector<String>();
        StringTokenizer st = new StringTokenizer(csvLine,separators,true);
        while (st.hasMoreTokens() ) {
            String element = st.nextToken();
            values.add(element);
        }

        boolean keepAdding;
        String[] elements = values.toArray(new String[values.size()]);
        if (expectedCount > 0 && expectedCount != elements.length)
        {
            //System.out.println("  ***  decodeCSVLine : comma separation correction for : " + csvLine);
//                debugOutputList(elements);
            for (int index = 0; index < (elements.length - 1); index++)
            {
                if (elements[index].startsWith("\"") && !elements[index].endsWith("\""))
                {
                    keepAdding = true;
                    for( int nextIndex = index; nextIndex < (elements.length - 1) && keepAdding; nextIndex++  )
                    {
                        elements[index] += ",";
                        elements[index] += elements[index + 1];
                        for (int nn = (index + 1); nn < elements.length; nn++)
                        {
                            if ((nn + 1) < elements.length)
                            {
                                elements[nn] = elements[(nn + 1)];
                            }
                            else
                            {
                                elements[nn] = "";
                            }
                        }
                        keepAdding = !elements[index].endsWith("\"");

//                        System.out.print("Elements : ");
//                        for (int pp = 0; pp < elements.length; pp++) {
//                            if ( pp > 0 ) System.out.print(",");
//                            System.out.print("[");
//                            System.out.print(elements[pp]);
//                            System.out.print("]");
//                        }
//                        System.out.println("");
                    }
                }
            }
        }
        return elements;
    }

    private String getTimingReport( String title, long startTime, long endTime, long numberOfActions ) {
        String reportTxt = "";
        String lineFeed = "\n";
        double durationSeconds = (double)(endTime - startTime) / 1000.0;
        reportTxt += lineFeed;
        reportTxt += "Title : " + title;
        reportTxt += lineFeed;
        reportTxt += "Start Time : " + ACUtilities.convertDateTimeToString(startTime);
        reportTxt += lineFeed;
        reportTxt += "End Time   : " + ACUtilities.convertDateTimeToString(endTime);
        reportTxt += lineFeed;
        reportTxt += "Duration Total : " + String.format("%.2f",durationSeconds) + " seconds";
        reportTxt += "  (";
        reportTxt += convertMillisecsToReadableDurationString((endTime - startTime));
        reportTxt += ")";
        reportTxt += lineFeed;
        if ( numberOfActions > 0 ) {
            reportTxt += "Number of Actions : " + numberOfActions;
            reportTxt += lineFeed;
            // Omitted : This info is not really an accurate guide to timing for single queries in an AbaReport
//            reportTxt += "Duration per Action : " +  String.format("%.3f",(durationSeconds/numberOfActions)) + " seconds";
//            reportTxt += lineFeed;
//            reportTxt += "Estimate for 10000 Actions : " + String.format("%.2f",((10000.0 * (double)durationSeconds/numberOfActions) / 60.0)) + " minutes";
//            reportTxt += lineFeed;
        }
        return reportTxt;
    }

    private String convertMillisecsToReadableDurationString(long durationInMilliseconds) {
        String formattedDuration = String.valueOf(durationInMilliseconds) + " ms";
        if ( durationInMilliseconds >= 0 ) {
            long hours = (long)((double)durationInMilliseconds / 1000.0 / 60.0 / 60.0);
            long minutes = (long)((double)(durationInMilliseconds - (hours * 60 * 60 * 1000)) / 1000.0 / 60.0);
            long seconds = (long)((double)(durationInMilliseconds - (hours * 60 * 60 * 1000) - (minutes * 60 * 1000)) / 1000.0);
            long milliseconds = durationInMilliseconds - (hours * 60 * 60 * 1000) - (minutes * 60 * 1000) - (seconds * 1000);
            long checkMillisecs = (hours * 60 * 60 * 1000) + (minutes * 60 * 1000) + (seconds * 1000) + milliseconds;
            StringBuilder sb = new StringBuilder();
            if ( hours > 0 ) {
                sb.append(String.format("%d hrs ", hours));
            }
            if ( minutes > 0 || sb.length() > 0 ) {
                sb.append(String.format("%d mins ", minutes));
            }
            if ( seconds > 0 || sb.length() > 0 ) {
                sb.append(String.format("%d secs ", seconds));
            }
            if ( milliseconds > 0 || sb.length() > 0 ) {
                sb.append(String.format("%d ms ", milliseconds));
            }
            sb.append(String.format("(%d ms)", durationInMilliseconds));
            if ( checkMillisecs != durationInMilliseconds ) {
                sb.append(String.format("  !!! check calc Diff(%d - %d = %d)", checkMillisecs, durationInMilliseconds, (checkMillisecs - durationInMilliseconds)));
            }
            formattedDuration = sb.toString();
        } else {
            formattedDuration += "  (negative)";
        }
        return formattedDuration;
    }

}
