/*
 * XmlFormatter.java  
 *
 * Creator:
 * 20.01.11 12:01 Sippel
 *
 * Maintainer:
 * 20.01.11 12:01 Sippel
 *
 * Last Modification:
 * $Id: XmlFormatter.java 109480 2017-03-03 10:18:53Z sippel $
 *
 * Copyright (c) 2003 ABACUS Research AG, All Rights Reserved
 */
package ch.abacus.xmltools;

import org.xml.sax.Attributes;

import java.util.ArrayList;

public class XmlFormatter extends TestSaxParser {

    private boolean mStripWhitespaces = false;
    private String mLineFeed = "\n";
    private String mIndent = "  ";
    private String mLastStartElementName = "";
    private StringBuffer mFormattedXml = new StringBuffer();

    ArrayList<String> mElementNames = new ArrayList<String>();

    private boolean mRemoveNamespacePrefixes = true;  // Default removes the namespaces from the Xml element names

    @Override
    public void startDocument() {
        super.startDocument();
        mFormattedXml.setLength(0);
    }

    public boolean isRemoveNamespacePrefixes() {
        return mRemoveNamespacePrefixes;
    }

    public void setRemoveNamespacePrefixes(boolean removeNamespacePrefixes) {
        mRemoveNamespacePrefixes = removeNamespacePrefixes;
    }

    public boolean isStripWhitespaces() {
        return mStripWhitespaces;
    }

    public void setStripWhitespaces(boolean stripWhitespaces) {
        mStripWhitespaces = stripWhitespaces;
    }

    public void setIndent(String singleIndent) {
        mIndent = singleIndent == null ? "" : singleIndent;
    }

    public void setLineFeed(String lineFeed) {
        mLineFeed = lineFeed == null ? "" : lineFeed;
    }

    @Override
    public void endElement(String name, String value) {
        mElementNames.remove(name);
        if ( name.equals(mLastStartElementName) ) {
            mLastStartElementName = "";
        } else {
            mFormattedXml.append(mLineFeed);
            int level = mElementNames.size();
            for ( int index = 0; index < level; index++ ) {
                mFormattedXml.append(mIndent);
            }
        }
        if ( ! mStripWhitespaces || ! isValueBlankLine(value) ) {
            mFormattedXml.append(escapeXMLText(value));
        }
        mFormattedXml.append("</");
        if ( mRemoveNamespacePrefixes ) {
            mFormattedXml.append(removeNamespacePrefix(name));
        } else {
            mFormattedXml.append(name);
        }
        mFormattedXml.append(">");
//        mFormattedXml.append(mLineFeed);
    }

    @Override
    public void startElement(String elementName, Attributes atts) {
        if ( mStripWhitespaces ) this.elementValue.reset();
//        mFormattedXml.append(mLineFeed);
        int level = mElementNames.size();
        if ( level > 0 /* && !"".equals(mLastStartElementName)  */) {
            mFormattedXml.append(mLineFeed);
        }
        for ( int index = 0; index < level; index++ ) {
            mFormattedXml.append(mIndent);
        }
        mElementNames.add(elementName);
        mFormattedXml.append("<");
        if ( mRemoveNamespacePrefixes ) {
            mFormattedXml.append(removeNamespacePrefix(elementName));
        } else {
            mFormattedXml.append(elementName);
        }
        if ( atts != null ) {
            int attLength = atts.getLength();
            if ( attLength > 0 ) {
                for ( int index = 0; index < attLength; index++ ) {
                    mFormattedXml.append(" ");
                    mFormattedXml.append(atts.getLocalName(index));
                    mFormattedXml.append("=");
                    mFormattedXml.append("\"");
                    mFormattedXml.append(atts.getValue(index));
                    mFormattedXml.append("\"");
                }
            }
        }
        mFormattedXml.append(">");

        mLastStartElementName = elementName;
    }

    public String getFormattedXml() {
        return mFormattedXml.toString();
    }

    private String removeNamespacePrefix(String elementName) {
        if ( elementName == null ) return elementName;
        int posColon = elementName.indexOf(":");
        return (posColon >= 0 ? elementName.substring(posColon+1) : elementName);
    }

    /**
     * Converts text with characters < > " & ' (ignores non ascii characters like linefeeds, etc)
     *   as :  &lt;  &gt; &quot;  &apos;
     *
     * @param text plain text
     * @return XML compatible text
     */
    private String escapeXMLText(String text) {
        String CODE_LESS_THAN       = "&lt;";
        String CODE_GREATER_THAN    = "&gt;";
        String CODE_QUOTE           = "&quot;";
        String CODE_AMP             = "&amp;";
        String CODE_APOSTROPHE      = "&apos;";
        String CODE_HEX_CHAR_PREFIX = "&#x";
        int ii = 0;

        StringBuilder sbXml = new StringBuilder(text);
        while(ii < sbXml.length()) {
            char ch = sbXml.charAt(ii);
            if(ch >= 32 && Character.isDefined(ch)) {
                switch(ch) {
                    case '\"':
                        sbXml.deleteCharAt(ii);
                        sbXml.insert(ii, CODE_QUOTE);
                        ii += CODE_QUOTE.length();
                        break;
                    case '&':
                        int totalLength = sbXml.length();
                        int subLength = ii+10;
                        String subText = (subLength >= totalLength ?
                                sbXml.substring(ii) : sbXml.substring(ii, ii + 10));
                        if ( !subText.startsWith(CODE_AMP) &&
                                !subText.startsWith(CODE_LESS_THAN) &&
                                !subText.startsWith(CODE_GREATER_THAN) &&
                                !subText.startsWith(CODE_QUOTE) && !subText.startsWith(CODE_APOSTROPHE)
                                && !subText.startsWith(CODE_HEX_CHAR_PREFIX) ) {
                            sbXml.deleteCharAt(ii);
                            sbXml.insert(ii, CODE_AMP);
                            ii += CODE_AMP.length();
                        } else {
                            ii++;
                        }
                        break;
                    case '\'':
                        sbXml.deleteCharAt(ii);
                        sbXml.insert(ii, CODE_APOSTROPHE);
                        ii += CODE_APOSTROPHE.length();
                        break;
                    case '<':
                        sbXml.deleteCharAt(ii);
                        sbXml.insert(ii, CODE_LESS_THAN);
                        ii += CODE_LESS_THAN.length();
                        break;
                    case '>':
                        sbXml.deleteCharAt(ii);
                        sbXml.insert(ii, CODE_GREATER_THAN);
                        ii += CODE_GREATER_THAN.length();
                        break;
                    default:
                        ii++;
                }
            } else {
                ii++;
            }
        }
        return sbXml.toString();
    }

}
