package ch.abacus.testapp;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.swing.*;
import javax.swing.table.DefaultTableModel;
import javax.swing.text.BadLocationException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.awt.event.*;
import java.awt.*;
import java.io.*;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Vector;


public class StandardExampleFrame extends JFrame {

    private static String EXAMPLE_CONFIG_FILENAME =  "abaengine_example_config.xml";
    private Rectangle mInitialFramePosition = null;

    private String mCurrentServerComputerName = "localhost";
    private long mCurrentPortNumber = 40000;

    private String mCurrentUserName = "Administrator";
    private String mCurrentUserPassword = "eli";
    private int mCurrentMandant = 7777;

    private String mCurrentInputParameters = "";
    private String mCurrentServerExportFileName = "";

    private int mButtonWidth = 230;
    private int mLinePanelHeight = 22;

    private JTextPane m_txpInfoBox = null;
    private JTextField m_txfServerComputerName = null;
    private JTextField m_txfPortNumber = null;
    private JTextField m_txfTimeout = null;
    private JCheckBox m_chkbSSLConnection = null;

    // This is the file name of the SSL certificates TrustStore, if used.  It will always be referenced to the actual
    // installed JAVA Path that is used at run-time (The JAVA path at runtime).
    private String mJavaTrustStoreCertificateFile = "private_certs";  // "private_certs" is a private file for certificates (not from JAVA)
    private String mKeystorePassword = "changeit";

    private int mTimeout = 500;  // 0.5 seconds for requests

    private JCheckBox m_chkbQueryAsView = null;
    private JCheckBox m_chkbShowLoginOnce = null;
    private JButton mBtnRunReport = null;
    private JComboBox<AbaViewReportType> m_cmbReportType = null;

    private ArrayList<String> mReportNamesList = new ArrayList<>();
    private String mSelectedReportName = "";

    private JCheckBox m_chkbLogSoapMessages = null;
/*

  Pdf("pdf", "pdf"),
  Text("txt", "txt"),
  PrintText("print-txt", "txt"),
  TextAll("txt_all", "txt"),
  PrintTextEndless("print-text_endless", "txt"),
  HtmlEndless("html_endless", "html"),
  HtmlTable("html_table", "html"),
  HtmlAutoTable("html_autotable", "html"),
  ExcelData("excel-xlsx_data", "xlsx"),
  ExcelMatrix("excel-xlsx_matrix", "xlsx"),
  Excel("excel-xlsx", "xls"),
  Word("word-docx", "docx"),
  XmlAttribute("xml_attr", "xml"),
  XmlValue("xml_value", "xml"),
  XmlNameValue("xml_namevalue", "xml"),
  XmlAv("xml_av", "xml"),
  XmlPreview("xml_preview", "xml");

Possibilities for outputFileType
txt :	TXT Datenexport (evtl. mit Codierung)
html_endless :	HTML Export
excel-xlsx :	Excel Dokument
word-docx :	Word Dokument
xml_value :	XML Dokument (evtl. mit Codierung)
pdf : 	PDF Dokument

*/
    private AbaViewReportType[] mReportTypeList = null;
    private AbaViewReportType mDefaultAbaViewReportType = null;

    private int mLoginCount = 0;

    class ReportTable extends JTable {
        public void changeSelection(int rowIndex, int columnIndex, boolean toggle, boolean extend) {
            tableCellSelectionChanged(rowIndex, columnIndex);
            super.changeSelection(rowIndex, columnIndex, toggle, extend);
        }
    }
    private ReportTable m_tblReportOutput = null;
    private JComboBox<String> m_cmbReportNames = null;
    private JTextField m_txfInputParameters = null;
    private JTextField m_txfServerExportFileName = null;

    private JLabel m_lblUrls = null;

    private String[] mUrls = null;

    private String[] mInterfaceNames = null;

    // Note : Since Abacus V2010 the retrieval of AbaReport names via the WebService is no longer available
    // Note : As of ABACUS V2016 and later it is not possible to query the report names list
    // The name of the report on the server must be known.  This functionality is obsolete

    private Vector mTableRowData = null;
    private Vector mTableHeaderData = null;

    private DefaultTableModel mTableModelData = new DefaultTableModel() {
        public int getRowCount() {
            if ( getTableRowData() == null ) return 3;
            int rows = getTableRowData().size();
            return (rows == 0 ? 3 : rows);
        }

        public int getColumnCount() {
            if ( mTableHeaderData == null ) return 5;
            int columns = mTableHeaderData.size();
            return (columns == 0 ? 5 : columns);
        }

        public Object getValueAt(int rowIndex, int columnIndex) {
            if ( getTableRowData() == null ) return null;
            int rowCount = getTableRowData().size();
            if ( rowIndex >= rowCount ) return null;
            Vector rowData = (Vector)getTableRowData().get(rowIndex);
            if ( rowData == null ) return null;
            int columnCount = rowData.size();
            if ( columnIndex >= columnCount ) return null;
            return rowData.get(columnIndex);
        }

        public String getColumnName(int column) {
            if ( mTableHeaderData == null ) return String.valueOf(column+1);
            int columnCount = mTableHeaderData.size();
            if ( column >= columnCount ) return String.valueOf(column+1);
            Object columnTitle = mTableHeaderData.get(column);
            return columnTitle == null ? "" : columnTitle.toString();
        }

        @Override
        public void setValueAt(Object aValue, int row, int column) {
            if ( getTableRowData() == null ) return;
            int rowCount = getTableRowData().size();
            if ( row >= rowCount ) return;
            Vector rowData = (Vector)getTableRowData().get(row);
            if ( rowData == null ) return;
            int columnCount = rowData.size();
            if ( column >= columnCount ) return;
            rowData.set(column, (aValue == null ? "": aValue));
        }
    };

    public static void main(String[] args) {
        final StandardExampleFrame frm = new StandardExampleFrame();
        frm.initUI();
        frm.setVisible(true);
    }

    public void tableCellSelectionChanged(int row, int column) {
        // Currently no functionality for table cell selection change
    }

    protected int getTimeoutMillisecs() {
        String millisecText = m_txfTimeout == null ? "" : m_txfTimeout.getText();
        mTimeout = m_txfTimeout == null ? 500 : ((int) convertTextToLong(millisecText, 500));
        if (mTimeout <= 0 || mTimeout > 9999999) mTimeout = 500; // Default to 500 ms
        if ( m_txfTimeout != null && !millisecText.equals(String.valueOf(mTimeout)) )
        {
            m_txfTimeout.setText(String.valueOf(mTimeout));
        }
        return mTimeout;
    }


    public Vector getTableHeaderData() {
        if ( mTableHeaderData == null ) mTableHeaderData = new Vector();
        return mTableHeaderData;
    }

    public Vector getTableRowData() {
        if ( mTableRowData == null ) mTableRowData = new Vector();
        return mTableRowData;
    }

    protected void initUI() {

        loadUserSettings();

        Container main = this.getContentPane();
//        JRootPane main = this.getRootPane();
        main.setLayout(new BorderLayout(5,5));
        main.add(getMainPanel());

        int desiredWidth = mInitialFramePosition == null ? 900 : mInitialFramePosition.width;
        int desiredHeight = mInitialFramePosition == null ? 600 : mInitialFramePosition.height;
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int width = (screenSize.width > desiredWidth ? desiredWidth:(screenSize.width - 50));
        int height = (screenSize.height > desiredHeight ? desiredHeight:(screenSize.height - 50));
        int xpos = mInitialFramePosition == null ? (screenSize.width - width) / 2 : mInitialFramePosition.x;
        int ypos = mInitialFramePosition == null ? (screenSize.height - height) / 2 : mInitialFramePosition.y;

        URL imageUrl = StandardExampleFrame.class.getResource("abaviewreport_icon.png");
        if ( imageUrl != null ) {
            ImageIcon mIcon = new ImageIcon(imageUrl);
            setIconImage(mIcon.getImage());
        }

        this.setSize(width, height);
        this.setLocation(xpos, ypos);

        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                closeConnection();
                dispose();
            }
        });
        mLoginCount = 0;
    }


    protected int getLoginCount() {
        return mLoginCount;
    }

    protected void incrementLoginCount() {
        mLoginCount++;
    }

    protected boolean isShowLoginOnceSelected() {
        return (m_chkbShowLoginOnce != null && m_chkbShowLoginOnce.isSelected());
    }

    protected int getCurrentMandant() {
        return mCurrentMandant;
    }

    protected void setCurrentMandant(int currentMandant) {
        mCurrentMandant = currentMandant;
    }

    protected String getCurrentUserName() {
        return mCurrentUserName;
    }

    protected void setCurrentUserName(String currentUserName) {
        mCurrentUserName = currentUserName;
    }

    protected String getCurrentUserPassword() {
        return mCurrentUserPassword;
    }

    protected void setCurrentUserPassword(String currentUserPassword) {
        mCurrentUserPassword = currentUserPassword;
    }

    public void setInterfaceNames(String[] interfaceNames) {
        mInterfaceNames = null;
        if ( interfaceNames == null ) return;
        if ( interfaceNames.length == 0 ) return;
        mInterfaceNames = new String[interfaceNames.length];
        System.arraycopy(interfaceNames, 0, mInterfaceNames, 0, interfaceNames.length);
        updateUrls();
    }

    private void updateUrls() {
        if ( mInterfaceNames == null ) return;
        if ( mInterfaceNames.length == 0 ) return;

        mLoginCount = 0;  // Reset login count if Url changes
        mUrls = new String[mInterfaceNames.length];
        for( int ii = 0; ii < mInterfaceNames.length; ii++ ) {
            mUrls[ii] = getDefaultUrl(getServerComputerName(), getPortNumber(),mInterfaceNames[ii], isSSLConnection());
        }
        fillUrlTextPane();
    }

    public boolean isSSLConnection() {
        return (m_chkbSSLConnection == null ? false : m_chkbSSLConnection.isSelected());
    }

    private void fillUrlTextPane() {
        if ( getUrlAddressLabel() == null ) return;

        if ( mUrls == null ) return;
        if ( mUrls.length == 0 ) return;

        StringBuilder sbUrls = new StringBuilder();
        for (String mUrl : mUrls) {
            if (sbUrls.length() > 0) sbUrls.append(("\n"));
            sbUrls.append(mUrl);
        }
        getUrlAddressLabel().setText(sbUrls.toString());

    }

    /**
     * Returns a URL String for the specified interface name.  The computer name is read from a
     * temporary file, if available, otherwise it is set to localhost.
     *
     * @param interfaceName the Interface name.
     * @return returns a default URL String for the specified interface name
     */
    private String getDefaultUrl(String serverComputerName, long portNumber, String interfaceName, boolean isSSLConnection)
    {
        String computerName = serverComputerName;
        long portNum = portNumber;
        //if ( portNum <= 0 ) portNum = 40000;
        if ( computerName == null || "".equals(computerName) ) computerName = getDefaultComputerName();
        String httpPrefix = isSSLConnection ? "https" : "http";
        return httpPrefix + "://" + computerName + (portNum > 0 ? (":" + portNum) : "") + "/." + (interfaceName == null ? "" : interfaceName);
    }

    private String getDefaultComputerName() {
        String defaultComputerName = "localhost";
        if ( !ACUtilities.StringIsNullOrEmpty(mCurrentServerComputerName) ) {
           defaultComputerName = mCurrentServerComputerName;
        }
        return defaultComputerName;
    }

    protected void closeConnection() {
        saveUserSettings();
    }

    private JPanel getMainPanel() {
        JPanel pnlMain = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.PAGE_AXIS,-1,-1);
        pnlMain.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));

        JPanel pnlComponents = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.PAGE_AXIS,-1,-1);

        pnlComponents.add(getComputerNamePortPanel());
        pnlComponents.add(Box.createVerticalStrut(6));
        pnlComponents.add(getUrlListPanel());
        pnlComponents.add(Box.createVerticalStrut(6));

        pnlComponents.add(getReportInputPanel());
        pnlComponents.add(Box.createVerticalStrut(4));
        pnlComponents.add(getReportParametersPanel());
        pnlComponents.add(Box.createVerticalStrut(4));
        pnlComponents.add(getExportFileNamePanel());
        pnlComponents.add(Box.createVerticalStrut(4));

        JSplitPane verticalSplitPane = new JSplitPane();
        verticalSplitPane.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 0));
        verticalSplitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
        verticalSplitPane.setResizeWeight(0.9);
        verticalSplitPane.setDividerSize(5);
        verticalSplitPane.setTopComponent(getReportOutputPanel());
        verticalSplitPane.setBottomComponent(getInfoPanel());
        verticalSplitPane.setAlignmentX(Component.CENTER_ALIGNMENT);
        verticalSplitPane.setDividerLocation(300);

        pnlComponents.add(verticalSplitPane);

        pnlMain.add(pnlComponents);

        return pnlMain;
    }

    private JPanel getComputerNamePortPanel() {
        JPanel pnlLine = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.LINE_AXIS,-1,mLinePanelHeight);

        pnlLine.add(UiComponentHelper.getLabel("Server Computer Name :", 150));
        m_txfServerComputerName = UiComponentHelper.getTextField("", 230);

        m_txfServerComputerName.setText(getDefaultComputerName());
        pnlLine.add(m_txfServerComputerName);

        pnlLine.add(Box.createHorizontalStrut(20));

        pnlLine.add(UiComponentHelper.getLabel("Port :", 40));
        m_txfPortNumber = UiComponentHelper.getTextField("", 60);
        m_txfPortNumber.setText(mCurrentPortNumber <= 0 ? "40000" : String.valueOf(mCurrentPortNumber));

        pnlLine.add(m_txfPortNumber);

        m_chkbSSLConnection = UiComponentHelper.getCheckBox("Use SSL", 80);
        m_chkbSSLConnection.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                action_UseSSLChanged();
            }
        });

        m_chkbShowLoginOnce = UiComponentHelper.getCheckBox("Show login dialog 1st time only", mButtonWidth);
        // Default is selected so that the login dialog will only be shown for the first login
        // For further queries the same login information will be used
        m_chkbShowLoginOnce.setSelected(true);

        pnlLine.add(Box.createHorizontalStrut(20));
        pnlLine.add(m_chkbSSLConnection);
        pnlLine.add(Box.createHorizontalGlue());
        pnlLine.add(m_chkbShowLoginOnce);

        m_txfServerComputerName.addKeyListener(new KeyListener() {
            public void keyPressed(KeyEvent e) {
            }
            public void keyReleased(KeyEvent e) {
                action_ComputerNameChanged();
            }
            public void keyTyped(KeyEvent e) {
            }
        });

        m_txfPortNumber.addKeyListener(new KeyListener() {
            public void keyPressed(KeyEvent e) {
            }
            public void keyReleased(KeyEvent e) {
                action_PortNumberChanged();
            }
            public void keyTyped(KeyEvent e) {
            }
        });

        m_txfServerComputerName.setToolTipText("<html>Server Computer Name</html>");
        m_txfPortNumber.setToolTipText("<html>Port Number</html>");
        m_chkbShowLoginOnce.setToolTipText("<html>Check to show the login dialog once or<br>uncheck to show login for every query.</html>");

        return pnlLine;
    }

    private JLabel getUrlAddressLabel() {
        if ( m_lblUrls == null ) {
            m_lblUrls = UiComponentHelper.getLabel("",-1);
        }
        return m_lblUrls;
    }

    private JPanel getUrlListPanel() {
        JPanel pnl = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.LINE_AXIS,-1,mLinePanelHeight);

        pnl.add(UiComponentHelper.getLabel("AbaEngine Service URL :", 150));
        pnl.add(getUrlAddressLabel());

        m_txfTimeout = UiComponentHelper.getTextField("", 50);
        m_txfTimeout.setText(String.valueOf(mTimeout));
        String labelTxt = "Timeout (ms) : ";
        int lblWidth = getFontMetricsStringWidth(labelTxt, 80);

        pnl.add(UiComponentHelper.getLabel(labelTxt, lblWidth));
        pnl.add(m_txfTimeout);
        pnl.add(Box.createHorizontalStrut(10));

        labelTxt = "Log Soap Messages";
        m_chkbLogSoapMessages = UiComponentHelper.getCheckBox(labelTxt, getFontMetricsStringWidth(labelTxt,120) + 25);
        pnl.add(Box.createHorizontalStrut(30));
        pnl.add(m_chkbLogSoapMessages);

        return pnl;
    }

    private JPanel getReportInputPanel() {
        JPanel pnl = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.LINE_AXIS,-1,mLinePanelHeight);

        pnl.add(UiComponentHelper.getLabel("AbaView Report Name :", 150));

        m_cmbReportNames = new JComboBox<>();
        m_cmbReportNames.setMinimumSize(new Dimension(10,20));
        m_cmbReportNames.setMaximumSize(new Dimension(9999, 20));
        m_cmbReportNames.setPreferredSize(new Dimension(500, 20));
        m_cmbReportNames.setEditable(true);
        m_cmbReportNames.setMaximumRowCount(20);

        mBtnRunReport = UiComponentHelper.getButton("Run Report",mButtonWidth);

        String labelTxt = "Use OpenView";
        m_chkbQueryAsView = UiComponentHelper.getCheckBox(labelTxt, getFontMetricsStringWidth(labelTxt,120) + 25);

        pnl.add(m_cmbReportNames);
        pnl.add(Box.createHorizontalStrut(10));
        pnl.add(m_chkbQueryAsView);
        pnl.add(Box.createHorizontalStrut(30));
        pnl.add(mBtnRunReport);

        m_cmbReportNames.setToolTipText("<html>AbaView Report Name can be typed or selected with the<br>mouse from the table when report names are shown.</html>");

        mBtnRunReport.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                action_runReport();
            }
        });

        if ( mReportNamesList != null && mReportNamesList.size() > 0 ) {
            m_cmbReportNames.setModel(new DefaultComboBoxModel<>(mReportNamesList.toArray(new String[0])));
            mBtnRunReport.setEnabled(true);
        } else {
            mBtnRunReport.setEnabled(false);
        }
        if ( !ACUtilities.StringIsNullOrEmpty(mSelectedReportName) ) {
            m_cmbReportNames.setSelectedItem(mSelectedReportName);
        }

        // Add a key listener to enable the RunReport button if the report name is typed manually
        m_cmbReportNames.getEditor().getEditorComponent().addKeyListener(new KeyListener() {
            @Override
            public void keyTyped(KeyEvent e) {
            }
            @Override
            public void keyPressed(KeyEvent e) {
            }
            @Override
            public void keyReleased(KeyEvent e) {
                mBtnRunReport.setEnabled(!"".equals(getReportName()));
            }
        });
        return pnl;
    }

    private int getFontMetricsStringWidth(String text, int defaultWidth) {
        int fontStringWidth = defaultWidth;
        if ( text != null && !"".equals(text) ) {
            fontStringWidth = getUrlAddressLabel().getFontMetrics(getUrlAddressLabel().getFont()).stringWidth(text);
        }
        return fontStringWidth;
    }

    /*
Information from : http://localhost:40000/av/abaengine/soap/api_runreport.html

Encoding 	Sprache
/ansi 	Windows Zeichensatz (ANSI, Windows-1252)
/ascii 	DOS Zeichensatz (ASCII, Codepage 858)
/UTF-8 	Unicode UTF-8
/UTF-16 	Unicode UTF-16
*/
    private AbaViewReportType[] getAbaReportTypes() {
        ArrayList<AbaViewReportType> reportTypes = new ArrayList<AbaViewReportType>();
        // mReportTypeList = new String[] {"txt","html","html_endless", "html_table","pdf", "excel-xlsx"};
        String[] encodings = new String[] {"ansi","ascii","UTF-8","UTF-16"};

        mDefaultAbaViewReportType = new AbaViewReportType("Text", "txt", "txt", encodings);   // Default
        reportTypes.add(mDefaultAbaViewReportType);

        // All Types from the ch.abacus.ulc.server.report.abaengine.AbaEngineOutputDevice Enum Class
        reportTypes.add(new AbaViewReportType("Pdf","pdf","pdf"));
        reportTypes.add(new AbaViewReportType("PrintText", "print-txt", "txt"));
        reportTypes.add(new AbaViewReportType("TextAll", "txt_all", "txt", encodings));
//        reportTypes.add(new AbaViewReportType("PrintTextEndless", "print-text_endless", "txt"));  // This is not supported
        reportTypes.add(new AbaViewReportType("HtmlEndless", "html_endless", "html"));
        reportTypes.add(new AbaViewReportType("HtmlTable", "html_table", "html"));
        reportTypes.add(new AbaViewReportType("HtmlAutoTable", "html_autotable", "html"));
        reportTypes.add(new AbaViewReportType("ExcelData", "excel-xlsx_data", "xlsx"));
        reportTypes.add(new AbaViewReportType("ExcelMatrix", "excel-xlsx_matrix", "xlsx"));
        reportTypes.add(new AbaViewReportType("Excel", "excel-xlsx", "xls"));
        reportTypes.add(new AbaViewReportType("Word", "word-docx", "docx"));
        reportTypes.add(new AbaViewReportType("XmlAttribute", "xml_attr", "xml", encodings));
        reportTypes.add(new AbaViewReportType("XmlValue", "xml_value", "xml", encodings));
        reportTypes.add(new AbaViewReportType("XmlNameValue", "xml_namevalue", "xml", encodings));
        reportTypes.add(new AbaViewReportType("XmlAv", "xml_av", "xml", encodings));
        reportTypes.add(new AbaViewReportType("XmlPreview", "xml_preview", "xml"));

        // Extra older types not defined above but listed in the older Documentation
        // These cannot be used with AbaEngineOutputDevice (e.g. Scripting)
        reportTypes.add(new AbaViewReportType("Excel", "excel", "xls"));
        reportTypes.add(new AbaViewReportType("ExcelData", "excel_data", "xls"));
        reportTypes.add(new AbaViewReportType("Word", "word", "docx"));
        reportTypes.add(new AbaViewReportType("XmlQuery", "xml_query", "xml"));
        reportTypes.add(new AbaViewReportType("Print", "print", "txt"));
//        reportTypes.add(new AbaViewReportType("Adf", "adf", "adf"));  // This is not supported

        Collections.sort(reportTypes);
        return reportTypes.toArray(new AbaViewReportType[0]);
    }

    public String getReportAbaEngineOutputDevice(String reportType) {
        String reportTypeExtension = "txt";
        if ( mReportTypeList == null || reportType == null || "".equals(reportType) ) return reportTypeExtension;
        if ( mReportTypeList.length > 0 ) {
            int matchCount = 0;
            for (AbaViewReportType avReportType : mReportTypeList ) {
                if ( reportType.equals(avReportType.getAbaReportType()) ) {
                    reportTypeExtension = avReportType.getAbaEngineOutputDevice();
                    matchCount++;
                }
            }
            if ( matchCount > 1 ) {
                throw new IllegalStateException("Multiple Report Types detected !!");
            }
        }
        return reportTypeExtension;
    }

    public String getReportFileExtension(String reportType) {
        String reportTypeExtension = "txt";
        if ( mReportTypeList == null || reportType == null || "".equals(reportType) ) return reportTypeExtension;
        if ( mReportTypeList.length > 0 ) {
            int matchCount = 0;
            for (AbaViewReportType avReportType : mReportTypeList ) {
                if ( reportType.equals(avReportType.getAbaReportType()) ) {
                    reportTypeExtension = avReportType.getFileExtension();
                    matchCount++;
                }
            }
            if ( matchCount > 1 ) {
                throw new IllegalStateException("Multiple Report Types detected !!");
            }
        }
        return reportTypeExtension;
    }

    public boolean getReportSupportsEncoding(String reportType, String encoding) {
        boolean supportsEncoding = false;
        if ( mReportTypeList == null || reportType == null || "".equals(reportType) ) return supportsEncoding;
        if ( mReportTypeList.length > 0 ) {
            int matchCount = 0;
            for (AbaViewReportType avReportType : mReportTypeList ) {
                if ( reportType.equals(avReportType.getAbaReportType()) ) {
                    supportsEncoding = avReportType.supportsEncoding(encoding);
                    matchCount++;
                }
            }
            if ( matchCount > 1 ) {
                throw new IllegalStateException("Multiple Report Types detected !!");
            }
        }
        return supportsEncoding;
    }

    private JPanel getReportParametersPanel() {
        JPanel pnl = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.LINE_AXIS,-1,mLinePanelHeight);

//        pnl.setAlignmentX(Component.CENTER_ALIGNMENT);

        pnl.add(UiComponentHelper.getLabel("Input Parameters :", 150));

        m_txfInputParameters = UiComponentHelper.getTextField("",0);

        m_txfInputParameters.setText(mCurrentInputParameters);

        mReportTypeList = getAbaReportTypes();

        int labelWidth = 90;
        JLabel lblReportType = UiComponentHelper.getLabel("Report Type :", labelWidth);
        m_cmbReportType = new JComboBox<>(mReportTypeList);
        int comboBoxWidth = mButtonWidth - labelWidth;
        m_cmbReportType.setMinimumSize(new Dimension(comboBoxWidth,20));
        m_cmbReportType.setMaximumSize(new Dimension(comboBoxWidth,20));
        m_cmbReportType.setPreferredSize(new Dimension(comboBoxWidth,20));
        m_cmbReportType.setMaximumRowCount(25);
        if ( mDefaultAbaViewReportType != null ) {
            m_cmbReportType.setSelectedItem(mDefaultAbaViewReportType);
        }

        pnl.add(m_txfInputParameters);
        pnl.add(Box.createHorizontalStrut(30));
        pnl.add(lblReportType);
//        pnl.add(Box.createHorizontalStrut(mButtonWidth));

        pnl.add(m_cmbReportType);

        m_txfInputParameters.setToolTipText("<html>Report input parameters separated by semi-colon<br>(e.g. FROM_DATE=01/01/2007;TO_DATE=01/10/2007 where<br> FROM_DATE and TO_DATE are the names of AbaView report input fields)</html>");
        return pnl;
    }

    public String getSelectedAbaViewReportType() {
        if ( m_cmbReportType == null ) return "txt";
        Object selectedItem = m_cmbReportType.getSelectedItem();
        return (selectedItem == null ? "txt" : selectedItem.toString());
    }

    private JPanel getExportFileNamePanel() {
        JPanel pnl = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.LINE_AXIS,-1,mLinePanelHeight);

//        pnl.setAlignmentX(Component.CENTER_ALIGNMENT);

        pnl.add(UiComponentHelper.getLabel("Server Export File :", 150));


        m_txfServerExportFileName = UiComponentHelper.getTextField("",0);

        m_txfServerExportFileName.setText(mCurrentServerExportFileName);

        pnl.add(m_txfServerExportFileName);
        pnl.add(Box.createHorizontalStrut(30));
        pnl.add(Box.createHorizontalStrut(mButtonWidth));

        m_txfServerExportFileName.setToolTipText("Path name of export file on the Server (e.g. c:\\test.txt)");
        return pnl;
    }


    private JPanel getReportOutputPanel() {
        JPanel pnl = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.LINE_AXIS,-1,-1);

        m_tblReportOutput = new ReportTable();

        JScrollPane scrPane = new JScrollPane();
        scrPane.setMaximumSize(new Dimension(9999,9999));
        scrPane.setMinimumSize(new Dimension(20,20));
        scrPane.setPreferredSize(new Dimension(1100,300));
        scrPane.setViewportView(m_tblReportOutput);

        m_tblReportOutput.setModel(mTableModelData);
        pnl.add(scrPane);

        return pnl;
    }

    private JPanel getInfoPanel() {
        JPanel pnlPage = UiComponentHelper.getPanelWithBoxLayout(BoxLayout.LINE_AXIS,-1,-1);

//        pnlPage.setBorder(BorderFactory.createEmptyBorder(0,0,0,10));

        m_txpInfoBox = new JTextPane();
        JScrollPane scrPane = new JScrollPane();
        scrPane.setMaximumSize(new Dimension(9999,9999));
        scrPane.setMinimumSize(new Dimension(20,20));
        scrPane.setPreferredSize(new Dimension(1100,300));
        scrPane.setViewportView(m_txpInfoBox);

        pnlPage.add(scrPane);

        m_txpInfoBox.setEditable(false);
        m_txpInfoBox.setBackground(new Color(250,250,250));
        return pnlPage;
    }

    private String getOutputPaneText() {
        return m_txpInfoBox == null ? "" : m_txpInfoBox.getText();
    }

    protected void writeToOutputPane( String text ) {
        if ( m_txpInfoBox != null ) {
            StringBuilder sb = new StringBuilder();
            String lineBreak = "\r\n";
            String currentText = m_txpInfoBox.getText();
            sb.append(currentText);
            if ( sb.length() > 0 && !currentText.endsWith(lineBreak) ) sb.append(lineBreak);
            sb.append(text);
            sb.append(lineBreak);
            m_txpInfoBox.setText(sb.toString());
            int iPos = m_txpInfoBox.getText().replace(lineBreak,"").length()-1;
            if ( iPos < 0 ) iPos = 0;
            try {
                m_txpInfoBox.scrollRectToVisible(m_txpInfoBox.modelToView(iPos));
                m_txpInfoBox.updateUI();
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        }
    }

    protected void scrollOutputPaneToStart() {
        if ( m_txpInfoBox != null ) {
            m_txpInfoBox.setCaretPosition(0);
            m_txpInfoBox.scrollRectToVisible(new Rectangle(0,0,2,2));
        }
    }

    public void refreshReportTable() {
        if ( m_tblReportOutput == null ) return;
        mTableModelData.fireTableDataChanged();
        mTableModelData.fireTableStructureChanged();
        m_tblReportOutput.updateUI();
    }

    protected void showMessageBox(String messageText) {
        JOptionPane.showMessageDialog(this,messageText );
    }

    protected void showMessageBox(String messageText, String title) {
        JOptionPane.showMessageDialog(this,messageText, title, JOptionPane.INFORMATION_MESSAGE );
    }

    protected int showConfirmMessageBox(String messageText, String title) {
        return showConfirmMessageBox(messageText, title, JOptionPane.OK_CANCEL_OPTION);
    }

    protected int showConfirmMessageBox(String messageText, String title, int optionPaneOption) {
        return JOptionPane.showConfirmDialog(this,messageText, title, optionPaneOption );
    }

    protected boolean isQueryAsOpenView() {
        if ( m_chkbQueryAsView == null ) return false;
        return m_chkbQueryAsView.isSelected();
    }

    protected boolean isLogSoapMessagesActive() {
        if ( m_chkbLogSoapMessages == null ) return false;
        return m_chkbLogSoapMessages.isSelected();
    }

    protected String getReportName() {
        String reportName = "";
        if ( m_cmbReportNames == null ) return reportName;
        ComboBoxEditor cmbEditor =  m_cmbReportNames.getEditor();
        if ( cmbEditor == null ) return reportName;
        Object item = cmbEditor.getItem();
        if ( item != null ) {
            reportName = item.toString();
        }
        return reportName;
    }

    protected void addReportName(String reportName) {
        if ( m_cmbReportNames == null || ACUtilities.StringIsNullOrEmpty(reportName) ) return;
        int itemCount = m_cmbReportNames.getItemCount();
        boolean nameAlreadyExists = false;
        for ( int index = 0; index < itemCount; index++ ) {
            if ( reportName.equals(m_cmbReportNames.getItemAt(index)) ) {
                nameAlreadyExists = true;
                break;
            }
        }
        if ( !nameAlreadyExists ) {
            m_cmbReportNames.addItem(reportName);
        }
    }

    private void saveUserSettings() {
        if ( m_cmbReportNames != null ) {
            StringBuilder sbFileContent = new StringBuilder();
            String lineFeed = ACUtilities.getOperatingSystemLineSeparator();
            String indent = "  ";

            sbFileContent.append("<?xml version='1.0' encoding='utf-8'?>");
            sbFileContent.append(lineFeed);
            sbFileContent.append("<AbaEngineExampleConfiguration>");
            sbFileContent.append(lineFeed);

            sbFileContent.append(indent);
            sbFileContent.append("<Settings>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<ServerName>");
            sbFileContent.append(getServerComputerName());
            sbFileContent.append("</ServerName>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<PortNumber>");
            sbFileContent.append(getPortNumber());
            sbFileContent.append("</PortNumber>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<Username>");
            sbFileContent.append(mCurrentUserName);
            sbFileContent.append("</Username>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            // Don't save the password
//            sbFileContent.append(indent);
//            sbFileContent.append(indent);
//            sbFileContent.append("<Password>");
//            sbFileContent.append(mCurrentUserPassword);
//            sbFileContent.append("</Password>");
//            sbFileContent.append(lineFeed);

            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<Mandant>");
            sbFileContent.append(mCurrentMandant);
            sbFileContent.append("</Mandant>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            mInitialFramePosition = this.getBounds();
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<FramePosition>");
            sbFileContent.append(String.format("%d,%d,%d,%d", mInitialFramePosition.x, mInitialFramePosition.y, mInitialFramePosition.width, mInitialFramePosition.height));
            sbFileContent.append("</FramePosition>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<TimeOutMillisecs>");
            sbFileContent.append(getTimeoutMillisecs());
            sbFileContent.append("</TimeOutMillisecs>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<LogSoapMessages>");
            sbFileContent.append(isLogSoapMessagesActive() ? "true" : "false");
            sbFileContent.append("</LogSoapMessages>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<InputParameters>");
            sbFileContent.append(getInputParameters());
            sbFileContent.append("</InputParameters>");
            sbFileContent.append(lineFeed);

            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append(indent);
            sbFileContent.append("<ServerExportFileName>");
            sbFileContent.append(getServerExportFileName());
            sbFileContent.append("</ServerExportFileName>");
            sbFileContent.append(lineFeed);


            sbFileContent.append(indent);
            sbFileContent.append("</Settings>");
            sbFileContent.append(lineFeed);


            // --------------------------------------------
            sbFileContent.append(indent);
            sbFileContent.append("<ReportNamesList>");
            sbFileContent.append(lineFeed);

            String currentReportName = getReportName();
            int itemCount = m_cmbReportNames.getItemCount();
            for (int index = 0; index < itemCount; index++) {
                Object reportName = m_cmbReportNames.getItemAt(index);
                if ( reportName != null ) {
                    sbFileContent.append(indent);
                    sbFileContent.append(indent);
                    sbFileContent.append("<ReportName");
                    if ( reportName.equals(currentReportName) ) {
                        sbFileContent.append(" selected=\"true\"");
                    }
                    sbFileContent.append(">");
                    sbFileContent.append(reportName);
                    sbFileContent.append("</ReportName>");
                    sbFileContent.append(lineFeed);
                }
            }
            sbFileContent.append(indent);
            sbFileContent.append("</ReportNamesList>");
            sbFileContent.append(lineFeed);

            sbFileContent.append("</AbaEngineExampleConfiguration>");
            sbFileContent.append(lineFeed);

            if ( sbFileContent.length() > 0 ) {
                try {
                    String reportListFileNames = ACUtilities.getCurrentUserWorkingDirectory() + File.separator + EXAMPLE_CONFIG_FILENAME;
                    BufferedOutputStream outs;
                    outs = new BufferedOutputStream(new FileOutputStream(reportListFileNames));
                    outs.write(sbFileContent.toString().getBytes(StandardCharsets.UTF_8));
                    outs.close();
                } catch (Exception ge) {
                    ge.printStackTrace();
                }
            }
        }

    }

    private void loadUserSettings() {
        String reportListFileNames = ACUtilities.getCurrentUserWorkingDirectory() + File.separator + EXAMPLE_CONFIG_FILENAME;
        if (new File(reportListFileNames).exists() )
        {
            try {

                DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
                DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
                org.w3c.dom.Document doc = dBuilder.parse(reportListFileNames);

                //optional, but recommended
                //read this - http://stackoverflow.com/questions/13786607/normalization-in-dom-parsing-with-java-how-does-it-work
                doc.getDocumentElement().normalize();

                //String lineFeed = ACUtilities.getOperatingSystemLineSeparator();
                String rootNodeName = doc.getDocumentElement().getNodeName();

                mSelectedReportName = "";
                if ("AbaEngineExampleConfiguration".equals(rootNodeName)) {
                    NodeList nList = doc.getElementsByTagName("ReportName");
                    for (int temp = 0; temp < nList.getLength(); temp++) {
                        Node nNode = nList.item(temp);
                        // System.out.println("\nCurrent Element : " + nNode.getNodeName());

                        if (nNode.getNodeType() == Node.ELEMENT_NODE) {
                            Element eElement = (Element) nNode;

                            String reportName = eElement.getTextContent();
                            if ( !ACUtilities.StringIsNullOrEmpty(reportName) ) {
                                if ( !mReportNamesList.contains(reportName) ) {
                                    mReportNamesList.add(reportName);
                                }
                                if ( eElement.hasAttribute("selected") ) {
                                    if ( "true".equalsIgnoreCase(eElement.getAttribute("selected")) ) {
                                        mSelectedReportName = reportName;   // Set the report as selected
                                    }
                                } else if ( ACUtilities.StringIsNullOrEmpty(mSelectedReportName) ) {
                                    mSelectedReportName = reportName;  // Set the first available report as selected if it is not already defined
                                }
                            }


                        }
                    }

                    nList = doc.getElementsByTagName("Settings");
                    for (int temp = 0; temp < nList.getLength(); temp++) {
                        Node nNode = nList.item(temp);
                        // System.out.println("\nCurrent Element : " + nNode.getNodeName());

                        if (nNode.getNodeType() == Node.ELEMENT_NODE) {
                            Element eElement = (Element) nNode;

                            String value = getTextContent(eElement, "ServerName");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {
                                setServerComputerName(value.trim());
                            }
                            value = getTextContent(eElement, "PortNumber");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {
                                setPortNumber( ACUtilities.convertTextToLong(value.trim(), 40000 ));
                            }
                            value = getTextContent(eElement, "Username");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {
                                mCurrentUserName = value.trim();
                            }
                            value = getTextContent(eElement, "Password");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {
                                mCurrentUserPassword = value.trim();
                            }
                            value = getTextContent(eElement, "Mandant");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {
                                mCurrentMandant = (int) ACUtilities.convertTextToLong(value.trim(), 7777);
                            }
                            value = getTextContent(eElement, "FramePosition");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {

                                String[] values = value.split(",");
                                if ( values.length > 3 ) {
                                    int xpos = (int) convertTextToLong(values[0], -1);
                                    int ypos = (int) convertTextToLong(values[1], -1);
                                    int desiredWidth = (int) convertTextToLong(values[2], -1);
                                    int desiredHeight = (int) convertTextToLong(values[3], -1);
                                    if (xpos >= 0 && ypos >= 0 && desiredHeight > 0 && desiredWidth > 0) {
                                        mInitialFramePosition = new Rectangle(xpos, ypos, desiredWidth, desiredHeight);
                                    }
                                }
                            }

                            value = getTextContent(eElement, "InputParameters");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {
                                mCurrentInputParameters = value.trim();
                            }
                            value = getTextContent(eElement, "ServerExportFileName");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {
                                mCurrentServerExportFileName = value.trim();
                            }
                            value = getTextContent(eElement, "TimeOutMillisecs");
                            if ( !ACUtilities.StringIsNullOrEmpty(value) ) {
                                mTimeout = (int) convertTextToLong(value.trim(), 500);
                            }
                        }
                    }
                }
            } catch (Exception ge) {
                ge.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
            }

        }
        if ( mReportNamesList.size() > 1 ) {
            Collections.sort(mReportNamesList);
        }
    }

    /**
     * Extracts the text content of an Element Node
     *
     * @param element the parent Element Node
     * @param elementTagName the Tag name of the child element in the parent node
     * @return the text content of the the specified element or an empty string if it does not exist
     */
    static private String getTextContent(Element element, String elementTagName) {
        String textContent = "";
        if ( element != null ) {
            NodeList nodelist = element.getElementsByTagName(elementTagName);
            if ( nodelist != null ) {
                if ( nodelist.getLength() > 0 ) {
                    Node node = nodelist.item(0);
                    if ( node != null ) {
                        textContent = node.getTextContent();
                    }
                }
            }
        }
        return textContent == null ? "" : textContent;
    }

    protected String getInputParameters() {
        return (m_txfInputParameters == null ? "" :  m_txfInputParameters.getText());
    }

    protected String getServerExportFileName() {
        String filename = (m_txfServerExportFileName == null ? "" :  m_txfServerExportFileName.getText());
        return filename.replace("/","\\");
    }

    protected void setReportName(String txtReportName) {
        if ( m_cmbReportNames != null ) {
            m_cmbReportNames.setSelectedItem((txtReportName == null ? "" : txtReportName));
        }
    }

    static long convertTextToLong(String text, long defaultValue)
    {
        long val;
        try
        {
            val = Long.parseLong(text);
        }
        catch (NumberFormatException nfe)
        {
            val = defaultValue;
        }
        return val;
    }

    static String removeQuotes(String value) {
        if ( value == null ) return "";
        String correctedValue = value.trim();
        if ( correctedValue.startsWith("\"") ) {
            correctedValue = correctedValue.substring(1);
        }
        if ( correctedValue.endsWith("\"") ) {
            if ( correctedValue.length() <= 1 ) {
                correctedValue = "";
            } else {
                correctedValue = correctedValue.substring(0,(correctedValue.length()-1));
            }
        }
        return correctedValue;
    }

    public String getAbaEngineUrl() {
        return getAbaEngineUrl(getServerComputerName(), getPortNumber());
    }

    private String getAbaEngineUrl(String serverComputerName, long portNumber)
    {
        String httpProtocol = "http";

        if ( isSSLConnection() ) {
            // Port 40001 ist the default AbaSiouxSecure port number - so set https:
            httpProtocol = "https";
            initializeCertificateTrustStore();
        }

        if (serverComputerName == null || "".equals(serverComputerName) )
        {
            return httpProtocol + "://localhost" + (portNumber > 0 ? (":" + portNumber) : "") + "/.abaengine";
        }
        else
        {
            return httpProtocol + "://" + serverComputerName + (portNumber > 0 ? (":" + portNumber) : "") + "/.abaengine";
        }


    }

    /**
     * This method is called automatically if an SSL connection is selected when using a default URL list
     */
    private void initializeCertificateTrustStore() {
        // Normally, an official SSL Certificate (e.g. from SwissSign) should already be in the Java trust "KeyStore", so it will
        // be automatically accepted.  If it is a special Certificate it can be added to the JAVA keystore manually with a
        // separate program at the keystore location for the Certificate can be referenced in the Properties (as shown below)

        String certificateFile = mJavaTrustStoreCertificateFile;
        // certificateFile = "cacerts";
        // To use SSL connections in JAVA the Certificate must be saved in the JAVA trust store
        // The Certificate can be saved from the internet browser and then added to the JAVA trustStore using another
        // program.
        String fileseparator = File.separator;
        String javaSecurityPath = System.getProperty("java.home") + fileseparator + "lib" + fileseparator + "security";
        String keystoreLocation = javaSecurityPath + fileseparator + certificateFile;
        if ( ! new File(keystoreLocation).exists() ) {
            // Default to the California Certificate file installed with JAVA (normally)
            certificateFile = "cacerts";
            keystoreLocation = javaSecurityPath + fileseparator + certificateFile;
        }
        if ( new File(keystoreLocation).exists() ) {
            String currentKeystoreLocation = System.getProperty("javax.net.ssl.trustStore");
            if (!keystoreLocation.equals(currentKeystoreLocation)) {
                System.setProperty("javax.net.ssl.trustStore", keystoreLocation);
                // The default password is "changeit".  Depending on the target computer, the default password may have been changed
                System.setProperty("javax.net.ssl.trustStorePassword", mKeystorePassword);
            }
            String ksMsg = "INFO : Java TrustStore using : " + keystoreLocation;
            if ( ! getOutputPaneText().contains(ksMsg) ) {
                writeToOutputPane("");
                writeToOutputPane(ksMsg);
            }
        } else {
            writeToOutputPane("WARNING : No JAVA TrustStore available in : " + javaSecurityPath);
            writeToOutputPane("WARNING : Looking for certificate KeyStore in : " + (javaSecurityPath + fileseparator + certificateFile));
            writeToOutputPane("             or otherwise in JAVA KeyStore in : " + keystoreLocation);

        }
    }


    private String getServerComputerName() {
        return (m_txfServerComputerName == null ? "" :  m_txfServerComputerName.getText());
    }

    public void setServerComputerName(String serverComputerName) {
        if ( m_txfServerComputerName != null ) {
            m_txfServerComputerName.setText((serverComputerName == null ? "localhost" : serverComputerName));
        }
        mCurrentServerComputerName = (serverComputerName == null ? "localhost" : serverComputerName);
    }

    private long getPortNumber() {
        String portNum = (m_txfPortNumber == null ? "" :  m_txfPortNumber.getText());
        if ( "".equals(portNum) ) {
            portNum = "40000";
            if ( m_txfPortNumber != null ) {
                m_txfPortNumber.setText(portNum);
            }
        }
        return convertTextToLong(portNum,40000);
    }

    public void setPortNumber(long portNumber) {
        if ( m_txfPortNumber != null ) {
            m_txfPortNumber.setText((portNumber <= 0 ? "40000" : String.valueOf(portNumber)));
        }
        mCurrentPortNumber = (portNumber <= 0 ? 40000 : portNumber);
    }

    public void action_runReport() {
        System.out.println("NOT IMPLEMENTED : StandardExampleFrame:action_runReport() called");
    }

    public void action_ComputerNameChanged() {
        String computerName = m_txfServerComputerName.getText();
        if ( "l".equals(computerName) ) {
            computerName = "localhost";
            m_txfServerComputerName.setText(computerName);
            m_txfServerComputerName.setSelectionStart(1);
        }
        updateUrls();
    }

    public void action_PortNumberChanged() {
        updateUrls();
    }

    public void clearInfo() {
        if ( m_txpInfoBox != null ) m_txpInfoBox.setText("");
    }

    private void action_UseSSLChanged() {
        updateUrls();

        if ( isSSLConnection() ) {
            String certificateFile = mJavaTrustStoreCertificateFile;
            String fileseparator = File.separator;
            String javaSecurityPath = System.getProperty("java.home") + fileseparator + "lib" + fileseparator + "security";
            String keystoreLocation = javaSecurityPath + fileseparator + certificateFile;
            if (!new File(keystoreLocation).exists()) {
                // Default to the California Certificate file installed with JAVA (normally)
                certificateFile = "cacerts";
                keystoreLocation = javaSecurityPath + fileseparator + certificateFile;
            }
            if (new File(keystoreLocation).exists()) {
                String ksMsg = "INFO : Java TrustStore using : " + keystoreLocation;
                if (!getOutputPaneText().contains(ksMsg)) {
                    writeToOutputPane("");
                    writeToOutputPane(ksMsg);
                }
            }
        }
    }

}
